import { fields, Record } from "@mail/model/export";
import { Deferred } from "@web/core/utils/concurrency";
import { rpc } from "@web/core/network/rpc";
import { compareDatetime, effectWithCleanup } from "@mail/utils/common/misc";

export class DiscussChannel extends Record {
    static _name = "discuss.channel";
    static _inherits = { "mail.thread": "thread" };
    static id = "id";

    static new() {
        const channel = super.new(...arguments);
        // Handles subscriptions for non-members. Subscriptions for channels
        // that the user is a member of are handled by
        // `ir_websocket@_build_bus_channel_list`.
        effectWithCleanup({
            effect(busChannel, busService) {
                if (busService && busChannel) {
                    busService.addChannel(busChannel);
                    return () => busService.deleteChannel(busChannel);
                }
            },
            dependencies: (channel) => [
                channel.shouldSubscribeToBusChannel && channel.busChannel,
                channel.store.env.services.bus_service,
            ],
            reactiveTargets: [channel],
        });
        return channel;
    }

    /**
     * Retrieve an existing channel from the store or fetch it if missing.
     *
     * @param {number} channel_id
     * @return {Promise<DiscussChannel>}
     */
    static getOrFetch(channel_id) {
        const channel = this.store["discuss.channel"].get(channel_id);
        if (channel?.fetchChannelInfoState === "fetched" || channel_id < 0) {
            return Promise.resolve(channel);
        }
        const fetchChannelInfoDeferred = this.store.channelIdsFetchingDeferred.get(channel_id);
        if (fetchChannelInfoDeferred) {
            return fetchChannelInfoDeferred;
        }
        const def = new Deferred();
        this.store.channelIdsFetchingDeferred.set(channel_id, def);
        this.store.fetchChannel(channel_id).then(
            () => {
                this.store.channelIdsFetchingDeferred.delete(channel_id);
                const channel = this.store["discuss.channel"].get(channel_id);
                if (channel?.exists()) {
                    channel.fetchChannelInfoState = "fetched";
                    def.resolve(channel);
                } else {
                    def.resolve();
                }
            },
            () => {
                this.store.channelIdsFetchingDeferred.delete(channel_id);
                const channel = this.store["discuss.channel"].get(channel_id);
                if (channel?.exists()) {
                    def.reject(channel);
                } else {
                    def.reject();
                }
            }
        );
        return def;
    }

    /** Equivalent to DiscussChannel._allow_invite_by_email */
    get allow_invite_by_email() {
        return (
            this.channel_type === "group" ||
            (this.channel_type === "channel" && !this.group_public_id)
        );
    }
    get allowDescriptionsTypes() {
        return ["channel", "group"];
    }
    get allowDescription() {
        return this.allowDescriptionsTypes.includes(this.channel_type);
    }
    get areAllMembersLoaded() {
        return this.member_count === this.channel_member_ids.length;
    }
    /** @type {"video_full_screen"|undefined} */
    default_display_mode;
    get typesAllowingCalls() {
        return ["chat", "channel", "group"];
    }
    get allowCalls() {
        return (
            !this.isTransient &&
            this.typesAllowingCalls.includes(this.channel_type) &&
            !this.correspondent?.persona.eq(this.store.odoobot)
        );
    }
    channel_member_ids = fields.Many("discuss.channel.member", {
        inverse: "channel_id",
        onDelete: (r) => r?.delete(),
        sort: (m1, m2) => m1.id - m2.id,
    });
    /** @type {"chat"|"channel"|"group"|"livechat"|"whatsapp"|"ai_chat"|"ai_composer"} */
    channel_type;
    chatWindow = fields.One("ChatWindow", {
        inverse: "channel",
    });
    get chatChannelTypes() {
        return ["chat", "group"];
    }
    /** @type {"not_fetched"|"pending"|"fetched"} */
    fetchMembersState = "not_fetched";
    /** @type {"not_fetched"|"fetching"|"fetched"} */
    fetchChannelInfoState = "not_fetched";
    get memberListTypes() {
        return ["channel", "group"];
    }
    get hasMemberList() {
        return this.memberListTypes.includes(this.channel_type);
    }
    last_interest_dt = fields.Datetime();
    lastInterestDt = fields.Datetime({
        /** @this {import("models").Thread} */
        compute() {
            return compareDatetime(this.self_member_id?.last_interest_dt, this.last_interest_dt) > 0
                ? this.self_member_id?.last_interest_dt
                : this.last_interest_dt;
        },
    });
    onlineMembers = fields.Many("discuss.channel.member", {
        /** @this {import("models").DiscussChannel} */
        compute() {
            return this.channel_member_ids
                .filter((member) => member.isOnline)
                .sort((m1, m2) => this.store.sortMembers(m1, m2)); // FIXME: sort are prone to infinite loop (see test "Display livechat custom name in typing status")
        },
    });
    hasOtherMembersTyping = fields.Attr(false, {
        /** @this {import("models").DiscussChannel} */
        compute() {
            return this.otherTypingMembers.length > 0;
        },
    });
    hasSeenFeature = fields.Attr(false, {
        /** @this {import("models").DiscussChannel} */
        compute() {
            return this.store.channel_types_with_seen_infos.includes(this.channel_type);
        },
    });
    /** @type {number} */
    id = fields.Attr(undefined, {
        onUpdate() {
            const busService = this.store.env.services.bus_service;
            if (!busService.isActive && !this.isTransient) {
                busService.start();
            }
        },
    });
    get invitationLink() {
        if (!this.uuid || this.channel_type === "chat") {
            return undefined;
        }
        return `${window.location.origin}/chat/${this.id}/${this.uuid}`;
    }
    invited_member_ids = fields.Many("discuss.channel.member");
    lastMessageSeenByAllId = fields.Attr(undefined, {
        /** @this {import("models").DiscussChannel} */
        compute() {
            if (!this.hasSeenFeature) {
                return;
            }
            return this.channel_member_ids.reduce((lastMessageSeenByAllId, member) => {
                if (member.notEq(this.self_member_id) && member.seen_message_id) {
                    return lastMessageSeenByAllId
                        ? Math.min(lastMessageSeenByAllId, member.seen_message_id.id)
                        : member.seen_message_id.id;
                } else {
                    return lastMessageSeenByAllId;
                }
            }, undefined);
        },
    });
    lastSelfMessageSeenByEveryone = fields.One("mail.message", {
        /** @this {import("models").DiscussChannel} */
        compute() {
            if (!this.lastMessageSeenByAllId) {
                return false;
            }
            let res;
            // starts from most recent persistent messages to find early
            for (let i = this.persistentMessages.length - 1; i >= 0; i--) {
                const message = this.persistentMessages[i];
                if (!message.isSelfAuthored) {
                    continue;
                }
                if (message.id > this.lastMessageSeenByAllId) {
                    continue;
                }
                res = message;
                break;
            }
            return res;
        },
    });
    /**
     * To be overridden.
     * The purpose is to exclude technical channel_member_ids like bots and avoid
     * "wrong" seen message indicator
     * @returns {import("models").ChannelMember[]}
     */
    get membersThatCanSeen() {
        return this.channel_member_ids;
    }
    /** @type {Number|undefined} */
    member_count;
    get shouldSubscribeToBusChannel() {
        return Boolean(
            !this.isTransient &&
                !this.self_member_id &&
                (this.isLocallyPinned || this.chatWindow?.isOpen)
        );
    }
    get isChatChannel() {
        return this.chatChannelTypes.includes(this.channel_type);
    }
    otherTypingMembers = fields.Many("discuss.channel.member", {
        /** @this {import("models").DiscussChannel} */
        compute() {
            return this.typingMembers.filter((member) => !member.persona?.eq(this.store.self));
        },
    });
    offlineMembers = fields.Many("discuss.channel.member", {
        /** @this {import("models").DiscussChannel} */
        compute() {
            return this._computeOfflineMembers().sort(
                (m1, m2) => this.store.sortMembers(m1, m2) // FIXME: sort are prone to infinite loop (see test "Display livechat custom name in typing status")
            );
        },
    });
    thread = fields.One("mail.thread", {
        compute() {
            return { id: this.id, model: "discuss.channel" };
        },
        inverse: "channel",
        onDelete: (r) => r?.delete(),
    });
    memberBusSubscription = fields.Attr(false, {
        /** @this {import("models").Thread} */
        compute() {
            return (
                this.self_member_id?.memberSince >= this.store.env.services.bus_service.startedAt
            );
        },
        onUpdate() {
            this.store.updateBusSubscription();
        },
    });
    typingMembers = fields.Many("discuss.channel.member", { inverse: "channelAsTyping" });
    get unknownMembersCount() {
        return (this.member_count ?? 0) - (this.channel_member_ids.length ?? 0);
    }

    delete() {
        this.chatWindow?.close();
        super.delete(...arguments);
    }

    async fetchChannelMembers() {
        if (this.fetchMembersState === "pending") {
            return;
        }
        const previousState = this.fetchMembersState;
        this.fetchMembersState = "pending";
        const known_member_ids = this.channel_member_ids.map((channelMember) => channelMember.id);
        let data;
        try {
            data = await rpc("/discuss/channel/members", {
                channel_id: this.id,
                known_member_ids: known_member_ids,
            });
        } catch (e) {
            this.fetchMembersState = previousState;
            throw e;
        }
        this.fetchMembersState = "fetched";
        this.store.insert(data);
    }

    async markAsFetched() {
        await this.store.env.services.orm.silent.call("discuss.channel", "channel_fetched", [
            [this.id],
        ]);
    }

    /**
     * @returns {boolean} true if the channel was opened, false otherwise
     */
    openChannel() {
        return false;
    }

    /** @param {string} name */
    async rename(name) {
        const newName = name.trim();
        if (
            newName !== this.displayName &&
            ((newName && this.channel?.channel_type === "channel") || this.channel?.isChatChannel)
        ) {
            if (["channel", "group"].includes(this.channel_type)) {
                this.name = newName;
                await this.store.env.services.orm.call(
                    "discuss.channel",
                    "channel_rename",
                    [[this.id]],
                    { name: newName }
                );
            } else if (this.supportsCustomChannelName) {
                if (this.self_member_id) {
                    this.self_member_id.custom_channel_name = newName;
                }
                await this.store.env.services.orm.call(
                    "discuss.channel",
                    "channel_set_custom_name",
                    [[this.id]],
                    { name: newName }
                );
            }
        }
    }

    /** @returns {import("models").ChannelMember[]} */
    _computeOfflineMembers() {
        return this.channel_member_ids.filter((member) => !member.isOnline);
    }
}

DiscussChannel.register();
