# -*- coding: utf-8 -*-
# Part of Odoo. See LICENSE file for full copyright and licensing details.
import lxml.html

from odoo.addons.sms_twilio.tests.common import MockSmsTwilioApi
from odoo.addons.test_mass_mailing.tests.common import TestMassMailCommon
from odoo.addons.test_mass_mailing.tests.common import TestMassSMSCommon
from odoo.tests.common import users, tagged
from odoo.tools import mute_logger


@tagged('mailing_manage')
class TestMailingTest(TestMassMailCommon):

    @classmethod
    def setUpClass(cls):
        super().setUpClass()
        cls.test_records = cls.env['mailing.test.blacklist'].create([
            {
                'email_from': f'test.mailing.{idx}@test.example.com',
                'name': f'Test Mailing {idx}',
                'user_id': cls.user_marketing.id,
            }
            for idx in range(5)
        ])
        cls.test_mailing_bl = cls.env['mailing.mailing'].create({
            'body_html': '<p>Hello <t t-out="object.name"/></p>',
            'mailing_domain': [('id', 'in', cls.test_records.ids)],
            'mailing_model_id': cls.env['ir.model']._get_id('mailing.test.blacklist'),
            'mailing_type': 'mail',
            'name': 'TestButton',
            'preview': 'Preview {{ object.name }}',
            'subject': 'Subject {{ object.name }}',
        })

    @users('user_marketing')
    @mute_logger('odoo.addons.mail.models.mail_render_mixin')
    def test_mailing_test_button(self):
        mailing = self.test_mailing_bl.with_env(self.env)
        mailing_test = self.env['mailing.mailing.test'].create({
            'email_to': 'test@test.com',
            'mass_mailing_id': mailing.id,
        })

        with self.mock_mail_gateway():
            mailing_test.send_mail_test()

        # not great but matches send_mail_test, maybe that should be a method
        # on mailing_test?
        record = self.env[mailing.mailing_model_real].search([], limit=1)
        first_child = lxml.html.fromstring(self._mails.pop()['body']).xpath('//body/*[1]')[0]
        self.assertEqual(first_child.tag, 'div')
        self.assertIn('display:none', first_child.get('style'),
                      "the preview node should be hidden")
        self.assertEqual(first_child.text.strip(), "Preview " + record.name,
                         "the preview node should contain the preview text")

        # Test if bad inline_template in the subject raises an error
        mailing.write({'subject': 'Subject {{ object.name_id.id }}'})
        with self.mock_mail_gateway(), self.assertRaises(Exception):
            mailing_test.send_mail_test()

        # Test if bad inline_template in the body raises an error
        mailing.write({
            'subject': 'Subject {{ object.name }}',
            'body_html': '<p>Hello <t t-out="object.name_id.id"/></p>',
        })
        with self.mock_mail_gateway(), self.assertRaises(Exception):
            mailing_test.send_mail_test()

        # Test if bad inline_template in the preview raises an error
        mailing.write({
            'body_html': '<p>Hello <t t-out="object.name"/></p>',
            'preview': 'Preview {{ object.name_id.id }}',
        })
        with self.mock_mail_gateway(), self.assertRaises(Exception):
            mailing_test.send_mail_test()

    @users('user_marketing')
    @mute_logger('odoo.addons.mail.models.mail_render_mixin')
    def test_mailing_test_button_links(self):
        """This tests that the link provided by the View in Browser snippet is correctly replaced
        when sending a test mailing while the Unsubscribe button's link isn't, to preserve the testing route
        /unsubscribe_from_list.
        This also checks that other links containing the /view route aren't replaced along the way.
        """
        mailing = self.test_mailing_bl.with_env(self.env)
        mailing_test = self.env['mailing.mailing.test'].create({
            'email_to': 'test@test.com',
            'mass_mailing_id': mailing.id,
        })
        # Test if link snippets are correctly converted
        mailing.write({
            'body_html':
                '''<p>
                Hello <a href="http://www.example.com/view">World<a/>
                    <div class="o_snippet_view_in_browser o_mail_snippet_general pt16 pb16" style="text-align: center; padding-left: 15px; padding-right: 15px;">
                        <a href="/view">
                            View Online
                        </a>
                    </div>
                    <div class="o_mail_footer_links">
                        <a role="button" href="/unsubscribe_from_list" class="btn btn-link">Unsubscribe</a>
                    </div>
                </p>''',
            'preview': 'Preview {{ object.name }}',
            'subject': 'Subject {{ object.name }}',
        })

        with self.mock_mail_gateway():
            mailing_test.send_mail_test()

        body_html = self._mails.pop()['body']
        self.assertIn(f'/mailing/{mailing.id}/view', body_html)  # Is replaced
        self.assertIn('/unsubscribe_from_list', body_html)  # Isn't replaced
        self.assertIn('http://www.example.com/view', body_html)  # Isn't replaced

    def test_mailing_test_equals_reality(self):
        """ Check that both test and real emails will format the qweb and inline
        placeholders correctly in body and subject. """
        mailing = self.test_mailing_bl.with_env(self.env)
        mailing.write({
            'body_html': '<p>Hello {{ object.name }} <t t-out="object.name"/></p>',
            'subject': 'Subject {{ object.name }} <t t-out="object.name"/>',
        })
        mailing_test = self.env['mailing.mailing.test'].create({
            'email_to': 'test@test.com',
            'mass_mailing_id': mailing.id,
        })

        with self.mock_mail_gateway():
            mailing_test.send_mail_test()

        expected_test_record = self.env[mailing.mailing_model_real].search([], limit=1)
        self.assertEqual(expected_test_record, self.test_records[0], 'Should take first found one')
        expected_subject = f'Subject {expected_test_record.name} <t t-out="object.name"/>'
        expected_body = 'Hello {{ object.name }}' + f' {expected_test_record.name}'

        self.assertSentEmail(self.env.user.partner_id, ['test@test.com'],
            subject='[TEST] %s' % expected_subject,
            body_content=expected_body)

        with self.mock_mail_gateway():
            # send the mailing
            mailing.action_launch()
            self.env.ref('mass_mailing.ir_cron_mass_mailing_queue').method_direct_trigger()

        self.assertSentEmail(
            self.env.user.partner_id,
            [expected_test_record.email_from],
            subject=expected_subject,
            body_content=expected_body,
        )

        self.assertEqual(
            self.env['mailing.mailing.test'].create({
                'mass_mailing_id': mailing.id,
            }).email_to,
            'test@test.com',
            "Should use the value of the previous record's email_to as default",
        )

        # Also test that related messages were properly deleted
        mailing.subject = 'Dummy Subject'

        with self.mock_mail_gateway(mail_unlink_sent=True):
            mailing_test.send_mail_test()

        test_subject = '[TEST] %s' % mailing.subject
        self.assertSentEmail(
            self.env.user.partner_id,
            ['test@test.com'],
            subject=test_subject,
        )
        self.assertFalse(self.env['mail.mail'].search([('subject', '=', test_subject)]))
        self.assertFalse(self.env['mail.message'].search([('subject', '=', test_subject)]))


@tagged('mailing_manage', 'twilio')
class TestMailingSMSTest(TestMassSMSCommon, MockSmsTwilioApi):

    @classmethod
    def setUpClass(cls):
        super().setUpClass()
        cls._setup_sms_twilio(cls.user_marketing.company_id)

    def test_mass_sms_test_button_twilio(self):
        """ Test the testing tool when twilio is activated on company """
        self._setup_sms_twilio(self.user_marketing.company_id)

        mailing = self.env['mailing.mailing'].create({
            'name': 'TestButton',
            'subject': 'Subject {{ object.name }}',
            'preview': 'Preview {{ object.name }}',
            'state': 'draft',
            'mailing_type': 'sms',
            'body_plaintext': 'Hello {{ object.name }}',
            'mailing_model_id': self.env['ir.model']._get('res.partner').id,
        })
        mailing_test = self.env['mailing.sms.test'].with_user(self.user_marketing).create({
            'numbers': '+32456001122',
            'mailing_id': mailing.id,
        })

        for error_type, exp_state, exp_msg in [
            (False, 'outgoing', '<ul><li>Test SMS successfully sent to +32456001122</li></ul>'),
            (
                'wrong_number_format', 'outgoing',  # not sure why outgoing but hey
                "<ul><li>Test SMS could not be sent to +32456001122: The number you're trying to reach is not correctly formatted</li></ul>"
            ),
        ]:
            with self.subTest(error_type=error_type):
                with self.with_user('user_marketing'):
                    with self.mock_sms_twilio_gateway(error_type=error_type):
                        mailing_test.action_send_sms()

                notification = mailing.message_ids[0]
                self.assertEqual(notification.body, exp_msg)
                self.assertSMS(
                    self.env["res.partner"], '+32456001122', exp_state,
                )
