# Part of Odoo. See LICENSE file for full copyright and licensing details.

from odoo import _, api, fields, models
from odoo.tools.urls import urljoin as url_join


class ResConfigSettings(models.TransientModel):
    _inherit = 'res.config.settings'

    # Groups
    group_show_uom_price = fields.Boolean(
        string="Base Unit Price",
        default=False,
        implied_group="website_sale.group_show_uom_price",
        group='base.group_user',
    )
    group_product_price_comparison = fields.Boolean(
        string="Comparison Price",
        implied_group="website_sale.group_product_price_comparison",
        group='base.group_user',
        help="Add a strikethrough price to your /shop and product pages for comparison purposes."
             "It will not be displayed if pricelists apply."
    )

    # Modules
    module_website_sale_autocomplete = fields.Boolean("Address Autocomplete")
    module_website_sale_collect = fields.Boolean("Click & Collect")

    # Website-dependent settings
    add_to_cart_action = fields.Selection(related='website_id.add_to_cart_action', readonly=False)
    cart_recovery_mail_template = fields.Many2one(
        related='website_id.cart_recovery_mail_template_id',
        readonly=False,
    )
    cart_abandoned_delay = fields.Float(
        related='website_id.cart_abandoned_delay',
        readonly=False,
    )
    send_abandoned_cart_email = fields.Boolean(
        string="Abandoned Email",
        related='website_id.send_abandoned_cart_email',
        readonly=False,
    )
    salesperson_id = fields.Many2one(
        related='website_id.salesperson_id',
        readonly=False,
    )
    salesteam_id = fields.Many2one(related='website_id.salesteam_id', readonly=False)
    website_sale_prevent_zero_price_sale = fields.Boolean(
        string="Prevent Sale of Zero Priced Product",
        related='website_id.prevent_zero_price_sale',
        readonly=False,
    )
    website_sale_contact_us_button_url = fields.Char(
        string="Button Url",
        related='website_id.contact_us_button_url',
        readonly=False,
    )
    show_line_subtotals_tax_selection = fields.Selection(
        related='website_id.show_line_subtotals_tax_selection',
        readonly=False,
    )
    confirmation_email_template_id = fields.Many2one(
        related='website_id.confirmation_email_template_id', readonly=False
    )

    # Additional settings
    account_on_checkout = fields.Selection(
        string="Customer Accounts",
        selection=[
            ("optional", "Optional"),
            ("disabled", "Disabled"),
            ("mandatory", "Mandatory"),
        ],
        compute="_compute_account_on_checkout",
        inverse="_inverse_account_on_checkout",
        readonly=False,
        required=True,
    )
    ecommerce_access = fields.Selection(
        related='website_id.ecommerce_access',
        readonly=False,
    )

    enabled_gmc_src = fields.Boolean(
        string="Google Merchant Center",
        related='website_id.enabled_gmc_src',
        readonly=False,
    )
    gmc_xml_url = fields.Char(compute='_compute_gmc_xml_url')

    #=== COMPUTE METHODS ===#

    @api.depends('website_id.account_on_checkout')
    def _compute_account_on_checkout(self):
        for record in self:
            record.account_on_checkout = record.website_id.account_on_checkout or 'disabled'

    @api.depends('website_domain')
    def _compute_gmc_xml_url(self):
        for config in self:
            # Uses `config.get_base_url()` which fallbacks to `web․base․url` if `website_domain` is
            # not set.
            config.gmc_xml_url = url_join(config.get_base_url(), '/gmc.xml')

    def _inverse_account_on_checkout(self):
        for record in self:
            if not record.website_id:
                continue
            record.website_id.account_on_checkout = record.account_on_checkout
            # account_on_checkout implies different values for `auth_signup_uninvited`
            if record.account_on_checkout in ['optional', 'mandatory']:
                record.website_id.auth_signup_uninvited = 'b2c'
            else:
                record.website_id.auth_signup_uninvited = 'b2b'

    #=== ACTION METHODS ===#

    def action_view_delivery_provider_modules(self):
        return self.env['delivery.carrier'].install_more_provider()

    @api.readonly
    def action_open_abandoned_cart_mail_template(self):
        return {
            'name': _("Customize Email Templates"),
            'type': 'ir.actions.act_window',
            'res_model': 'mail.template',
            'view_id': False,
            'view_mode': 'form',
            'res_id': self.env['ir.model.data']._xmlid_to_res_id("website_sale.mail_template_sale_cart_recovery"),
        }

    def action_open_extra_info(self):
        self.ensure_one()
        # Add the "edit" parameter in the url to tell the controller
        # that we want to edit even if we are not in a payment flow
        return self.env["website"].get_client_action(
            '/shop/extra_info?open_editor=true', mode_edit=True, website_id=self.website_id.id)

    @api.readonly
    def action_open_sale_mail_templates(self):
        return {
            'name': _("Customize Email Templates"),
            'type': 'ir.actions.act_window',
            'domain': [('model', '=', 'sale.order')],
            'res_model': 'mail.template',
            'view_id': False,
            'view_mode': 'list,form',
        }
