import { SCHEME_TEMPLATE, FIELD_TEMPLATE, PARTITION } from "@barcodes_gs1_nomenclature/js/epc_model";
import { FNC1_CHAR } from "@barcodes_gs1_nomenclature/js/barcode_parser";

const REGEX_IS_HEXA = /^[0-9a-f]+$/i;

//  Technical Utilities
function getBitLength(num) {
    if (num === 0n) {
        return 1;
    } // Special case for 0
    let bitLength = 0;
    while (num > 0n) {
        num >>= 1n;
        bitLength++;
    }
    return bitLength;
}

function readBits(value, index, length, bitLength = null) {
    if (index < 0 || length < 0) {
        throw new Error("Index and length must be non-negative");
    }
    if (bitLength === null) {
        bitLength = getBitLength(value);
    }
    const indexEnd = index + length;
    if (indexEnd > bitLength) {
        throw new Error("Index range exceeds bit length of the value");
    }
    let read = value >> BigInt(bitLength - indexEnd); // Remove the rightmost unwanted part
    const mask = (1n << BigInt(length)) - 1n; // And Mask to extract the wanted part
    read &= mask;
    return read;
}

// EPC encoding must have a multiple of 16 bits.
// c.f. [TDS  2.1], §15.1.1
function missingBits(bitLength, standardSize = 16) {
    const remainder = bitLength % standardSize;
    return remainder === 0 ? 0 : standardSize - remainder;
}

// Decode Methods
function decodeField(encoding, model, fieldName) {
    switch (encoding) {
        case "blank":
            return null;
        case "integer":
            return decodeInteger(model, fieldName);
        case "partition_table":
            return decodePartition(model, fieldName);
        case "string":
            return decodeString(model, fieldName);
        default:
            throw new Error(`No decode function found for encoding: ${encoding}`);
    }
}

function decodeInteger(model, fieldName) {
    const field = model.fields[fieldName];
    const value = readBits(model.hexValue, field.offset, field.bitSize, model.bitLength);
    let res = value.toString();
    //Improve later
    if (field.digits) {
        res = res.padStart(field.digits, "0");
    }
    return res;
}

function decodePartition(model, fieldName) {
    const field = model.fields[fieldName];
    const partition = PARTITION[model.partition];
    if (partition === undefined) {
        throw new Error(`No partition found for value: ${model.partition}`);
    }
    const value = readBits(model.hexValue, field.offset, field.bitSize, model.bitLength);
    const fieldEntries = Object.entries(model.fields);
    const fieldIndex = fieldEntries.findIndex(([name]) => name === fieldName);
    if (fieldIndex === -1 || fieldIndex + 2 >= fieldEntries.length) {
        throw new Error(`Field ${fieldName} or subsequent fields are not defined.`);
    }
    // Update the bitSize of the next two fields based on partition information
    const [nextFieldName1] = fieldEntries[fieldIndex + 1];
    const [nextFieldName2] = fieldEntries[fieldIndex + 2];
    model.fields[nextFieldName1].bitSize = partition[value].left_bit;
    model.fields[nextFieldName1].digits = partition[value].left_digit;
    model.fields[nextFieldName2].bitSize = partition[value].right_bit;
    model.fields[nextFieldName2].digits = partition[value].right_digit;

    return null;
}

function decodeString(model, fieldName) {
    const field = model.fields[fieldName];
    const value = readBits(model.hexValue, field.offset, field.bitSize, model.bitLength);
    const charLength = field.bitSize / 7;
    const charCodes = [];
    // loop read 7 bits at a time
    for (let i = 0; i < charLength; i++) {
        const res = Number(readBits(value, i * 7, 7, field.bitSize));
        if (res === 0) {
            break;
        }
        charCodes[i] = res;
    }
    // parse to UTF string
    return String.fromCharCode(...charCodes);
}

// 'Business' Utilities
function toPureUri(model) {
    const fieldEntries = Object.entries(model.fields);
    const [uriHeader, uriBody] = getSplitUri(model.template.uri_pure_tag);
    for (let i = 0; i < fieldEntries.length; i++) {
        const index = uriBody.indexOf(fieldEntries[i][1].uriPortion);
        if (index !== -1) {
            uriBody[index] = fieldEntries[i][1].value;
            if (fieldEntries[i][1].encoding == "string") {
                uriBody[index] = encodeURIComponent(uriBody[index]); // For an URI, characters must be escaped
            }
        }
    }
    return uriHeader + ":" + uriBody.join(".");
}

function getSplitUri(uri) {
    const uriHeader = uri.split(":");
    const uriBody = uriHeader.pop();
    return [uriHeader.join(":"), uriBody.split(".")];
}

function toElementString(model) {
    const aiDict = model.template.ai_list;
    let elementString = "";
    let value;
    for (const [key, field] of Object.entries(aiDict)) {
        if (field === null) {
            value = getAi(key, model);
        } else {
            value = model.fields[field].value;
        }
        elementString += `${FNC1_CHAR}${key.padStart(2, "0")}${value}`;
    }
    return elementString;
}

function getAi(identifier, model) {
    switch (identifier) {
        case "0":
            return getAi00(model);
        case "1":
            return getAi01(model);
        case "414":
            return getAi414(model);
        default:
            throw new Error(
                `No extrapolation function found for the GS1 Application Identifier (${identifier.padStart(2, "0")})`
            );
    }
}

function getAi00(model) {
    const companyPrefix = model.fields["company_prefix"];
    const serialInteger = model.fields["serial_integer"];

    const extensionDigit = serialInteger.value.substr(0, 1);
    const serialRefRemainder = serialInteger.value.substr(1);
    const ssccToCheck = extensionDigit + companyPrefix.value + serialRefRemainder;
    const checkdigit = getGs1Checksum(ssccToCheck);
    return ssccToCheck + checkdigit;
}

function getAi01(model) {
    // input length : sum of companyPrefix.digits and itemReference.digits is always 13 digits (preserving leading zeroes)
    // output: 14 digits
    const companyPrefix = model.fields["company_prefix"];
    const itemReference = model.fields["item_reference"];

    const indicator = itemReference.value.substr(0, 1);
    const itemRefRemainder = itemReference.value.substr(1);
    const gtinToCheck = indicator + companyPrefix.value + itemRefRemainder;
    const checkdigit = getGs1Checksum(gtinToCheck);
    return gtinToCheck + checkdigit;
}

function getAi414(model) {
    const companyPrefix = model.fields["company_prefix"];
    const locationReference = model.fields["location_reference"];

    // company_prefix + location_ref
    const glnToCheck = companyPrefix.value + locationReference.value;
    const checkdigit = getGs1Checksum(glnToCheck);
    return glnToCheck + checkdigit;
}

function getGs1Checksum(data) {
    // The GS1 specification defines the check digit calculation as follows (cf. [TDS 2.1] §7.3):
    // d14 = (10 - ((3(d1 + d3 + d5 + d7 + d9 + d11 + d13) + (d2 + d4 + d6 + d8 + d10 + d12)) mod 10) mod 10
    // However, it should be noted that here the index starts at 0 and not 1.
    // It is therefore necessary to multiply even numbers instead of odd numbers.

    // Note : same method is used for multiple data structures up to 17 digits (excluding check digits).
    // Cf. https://ref.gs1.org/standards/genspecs/ §7.9.1
    // data = data.padStart(17, '0'); is not optimal so we cheat on i value of incoming for loop instead
    const offset = 17 - data.length;
    let sum = 0;
    for (let i = 0; i < data.length; i++) {
        const value = parseInt(data.charAt(i));
        if ((i + offset) % 2 == 0) {
            sum += value * 3;
        } else {
            sum += value;
        }
    }
    return ((10 - (sum % 10)) % 10).toString();
}

//public
export function isValidEpcFormat(hexString) {
    if (hexString.length < 2) {
        return false;
    }
    const headerString = hexString.substring(0,2);
    if (!REGEX_IS_HEXA.test(headerString)) {
        return false;
    }
    const header = parseInt(headerString, 16);
    return SCHEME_TEMPLATE[header]?.length === hexString.length //TODO: allow length in range for variable length schemes
        && REGEX_IS_HEXA.test(hexString);
}

export function decode(hexString, asURI = false) {
    // Prepare values
    let hexValue;
    try{
        hexValue = BigInt("0x" + hexString);
    }catch{
        return hexString; // return original value if unable to decode
    }
    const bitLength = getBitLength(hexValue);
    const headerRealBitCount = 8 - missingBits(bitLength, 8); // compensate for leftmost missing 0
    const header = Number(readBits(hexValue, 0, headerRealBitCount, bitLength));

    //Find template
    if (SCHEME_TEMPLATE[header] === undefined) {
        return hexString; // return original value if unable to decode
    }
    const template = SCHEME_TEMPLATE[header];
    // Create & initialize base scheme structure
    const model = {
        hexValue: hexValue,
        bitLength: bitLength,
        partition: template.partition,
        fields: {},
        template: template,
    };
    for (let i = 0; i < template.fields_template.length; i++) {
        const fieldName = template.fields_template[i];
        const fieldTemplate = FIELD_TEMPLATE[fieldName];
        const fieldBitCount = template.fields_bit_count[i];

        model.fields[fieldName] = {
            encoding: fieldTemplate.encoding,
            bitSize: i === 0 ? headerRealBitCount : fieldBitCount,
            uriPortion: fieldTemplate.uri_portion,
            offset: i === 0 ? 0 : null,
            value: i === 0 ? header.toString() : null,
        };
    }

    // Effectively process fields
    const fieldEntries = Object.entries(model.fields);
    for (let i = 1; i < fieldEntries.length; i++) {
        const [fieldName, fieldInfo] = fieldEntries[i];
        const previousField = fieldEntries[i - 1][1];
        fieldInfo.offset = previousField.offset + previousField.bitSize;
        fieldInfo.value = decodeField(fieldInfo.encoding, model, fieldName);
    }

    // return GS1 AI(s) as String
    if (asURI) {
        return toPureUri(model); // TODO : Should be the only available option for GID-96, USDOD-96 and ADI-var. Cf. [TDT 2.0] Figure 1-1
    }
    return toElementString(model);
}
