# Part of Odoo. See LICENSE file for full copyright and licensing details.

from odoo.addons.purchase_alternative.tests.common import TestPurchaseAlternativeCommon
from odoo.tests import Form, tagged
from odoo import Command


@tagged('at_install', '-post_install')  # LEGACY at_install
class TestPurchaseAlternativeStock(TestPurchaseAlternativeCommon):

    def test_04_purchase_alternative_stock(self):
        """Check that alt PO correctly copies the original PO values"""
        # create original PO
        orig_po = self.env['purchase.order'].create({
            'partner_id': self.res_partner_1.id,
            'picking_type_id': self.env['stock.picking.type'].search([['code', '=', 'outgoing']], limit=1).id,
            'dest_address_id': self.env['res.partner'].create({'name': 'delivery_partner'}).id,
        })
        unit_price = 50
        po_form = Form(orig_po)
        with po_form.order_line.new() as line:
            line.product_id = self.product_09
            line.product_qty = 5.0
            line.price_unit = unit_price
        po_form.save()

        # create an alt PO
        action = orig_po.action_create_alternative()
        alt_po_wiz = Form(self.env['purchase.alternative.create'].with_context(**action['context']))
        alt_po_wiz.partner_ids = self.res_partner_1
        alt_po_wiz.copy_products = True
        alt_po_wiz = alt_po_wiz.save()
        alt_po_wiz.action_create_alternative()

        # check alt PO was created with correct values
        alt_po = orig_po.alternative_po_ids.filtered(lambda po: po.id != orig_po.id)
        self.assertEqual(orig_po.picking_type_id, alt_po.picking_type_id,
                         "Alternative PO should have copied the picking type from original PO")
        self.assertEqual(orig_po.dest_address_id, alt_po.dest_address_id,
                         "Alternative PO should have copied the destination address from original PO")
        self.assertEqual(orig_po.order_line.product_id, alt_po.order_line.product_id,
                         "Alternative PO should have copied the product to purchase from original PO")
        self.assertEqual(orig_po.order_line.product_qty, alt_po.order_line.product_qty,
                         "Alternative PO should have copied the qty to purchase from original PO")
        self.assertEqual(len(alt_po.alternative_po_ids), 2,
                         "Newly created PO should be auto-linked to itself and original PO")

        # confirm the alt PO, original PO should be cancelled
        action = alt_po.button_confirm()
        warning_wiz = Form(
            self.env['purchase.alternative.warning'].with_context(**action['context']))
        warning_wiz = warning_wiz.save()
        self.assertEqual(len(warning_wiz.alternative_po_ids), 1,
                         "POs not in a RFQ status should not be listed as possible to cancel")
        warning_wiz.action_cancel_alternatives()
        self.assertEqual(orig_po.state, 'cancel', "Original PO should have been cancelled")

    def test_05_move_dest_links_alternatives(self):
        """ Checks if Purchase Order alternatives are correctly linked to the original PO's move_dest_ids. """
        # Prepare test data
        wh = self.env.ref('stock.warehouse0')
        buy_route_id = self.ref('purchase_stock.route_warehouse0_buy')
        vendor_1 = self.env['res.partner'].create({'name': 'Vendor 1'})
        vendor_2 = self.env['res.partner'].create({'name': 'Vendor 2'})
        product = self.env['product.product'].create({
            'name': 'Test product',
            'is_storable': True,
            'seller_ids': [Command.create({
                'partner_id': vendor_1.id,
                'price': 10.0,
                'delay': 0,
            })],
            'route_ids': [Command.set([buy_route_id])],
        })

        # Sets the warehouse to do two-steps receptions
        grp_multi_loc = self.env.ref('stock.group_stock_multi_locations')
        grp_multi_step_rule = self.env.ref('stock.group_adv_location')
        self.env.user.write({'group_ids': [(3, grp_multi_loc.id)]})
        self.env.user.write({'group_ids': [(3, grp_multi_step_rule.id)]})
        wh.reception_steps = 'two_steps'

        # Create a reordering rule for the product and
        self.env['stock.warehouse.orderpoint'].create({
            'name': 'RR for %s' % product.name,
            'warehouse_id': wh.id,
            'location_id': wh.lot_stock_id.id,
            'product_id': product.id,
            'product_min_qty': 1,
            'product_max_qty': 10,
        })
        # Run scheduler to create internal transfer from Input -> Stock and generate the Purchase Order
        self.env['stock.rule'].run_scheduler()
        # The internal move (Input -> Stock) shouldn't have been generated yet
        int_move = self.env['stock.move'].search([('product_id', '=', product.id)])
        self.assertFalse(int_move)
        # Find the Purchase Order generated by the orderpoint
        orig_po = self.env['purchase.order'].search([('partner_id', '=', vendor_1.id)])
        self.assertEqual(len(orig_po.ids), 1, "Only one PO should have been generated.")
        # Create an alternative RFQ for another vendor
        action = orig_po.action_create_alternative()
        alt_po_wizard = Form(self.env['purchase.alternative.create'].with_context(**action['context']))
        alt_po_wizard.partner_ids = vendor_2
        alt_po_wizard.copy_products = True
        alt_po_wizard = alt_po_wizard.save()
        alt_po_wizard.action_create_alternative()
        # Find the alternative Purchase Order
        alt_po = orig_po.alternative_po_ids.filtered(lambda po: po.id != orig_po.id)
        self.assertEqual(len(orig_po.alternative_po_ids), 2, "Base PO should be linked with the alternative PO.")
        # Validate it
        warning_wizard = Form.from_action(self.env, alt_po.button_confirm()).save()
        # Cancel other alternatives
        warning_wizard.action_cancel_alternatives()
        self.assertEqual(orig_po.state, 'cancel', "Original PO should have been cancelled.")
        self.assertEqual(alt_po.state, 'purchase', "Alternative PO should have been confirmed.")
        # Set all qty as done and validate the alternative PO's picking
        in_picking = alt_po.picking_ids
        self.assertEqual(in_picking.picking_type_id.code, 'incoming', "Must be the reception picking.")
        in_picking.move_ids.quantity = 10
        in_picking.move_ids.picked = True
        in_picking.button_validate()
        # Now the internal move (Input -> Stock) should be generated
        int_move = self.env['stock.move'].search([('product_id', '=', product.id), ('location_dest_id', '=', wh.lot_stock_id.id)])
        self.assertEqual(int_move.quantity, 10, "Quantity should be reserved in the original internal move.")
        self.assertEqual(int_move.move_orig_ids.id, in_picking.move_ids.id, "Both moves should be correctly chained together.")

    def test_group_id_alternative_po(self):
        """ Check that the group_id is propagated in the alternative PO"""
        orig_po = self.env['purchase.order'].create({
            'partner_id': self.res_partner_1.id,
        })
        # Creates an alternative PO
        action = orig_po.action_create_alternative()
        alt_po_wizard_form = Form(self.env['purchase.alternative.create'].with_context(**action['context']))
        alt_po_wizard_form.partner_ids = self.res_partner_1
        alt_po_wizard_form.copy_products = True
        alt_po_wizard = alt_po_wizard_form.save()
        alt_po_id = alt_po_wizard.action_create_alternative()['res_id']
        alt_po = self.env['purchase.order'].browse(alt_po_id)
        self.assertEqual(alt_po.reference_ids, orig_po.reference_ids)
